/******************** (C) COPYRIGHT 2008 STMicroelectronics ********************
* File Name          : main.c
* Author             : MCD Application Team
* Version            : V2.0.1
* Date               : 06/13/2008
* Description        : Main program body
********************************************************************************
* THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
* WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
* AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT,
* INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE
* CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING
* INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/

// Interface LCD Nikia-5110 By SPI Function
// STM32F103RBT6    <--->  LCD Nokia-5110 
// PB.8(GPIO:OUT)   ---->  SCE(Chips Select Active "0")
// PB.9(GPIO:OUT)   ---->  RES(Reset Active "0")
// PB.10(GPIO:OUT)  ---->  D/C("1"=Data/"0"=Command)
// PB.11(GPIO:OUT)  ---->  LED(Backlight)

// PB.13(SPI2:SCK)  ---->  SCLK
// PB.15(SPI2:MOSI) ---->  SDIN	

/* Includes ------------------------------------------------------------------*/
#include "stm32f10x_lib.h"
#include "platform_config.h"

/* Define For NOKIA-5110 */
#include "font.h"	   								// Character Font Code
#include "picture.h"								// Bitmap Picture
/* End Define NOKIA-5110 */

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
GPIO_InitTypeDef GPIO_InitStructure;
ErrorStatus HSEStartUpStatus;
SPI_InitTypeDef   SPI_InitStructure;

/* Private function prototypes -----------------------------------------------*/
void RCC_Configuration(void);
void NVIC_Configuration(void);
void lcd_write_data(unsigned char DataByte);					// Write Data to LCD
void lcd_write_command(unsigned char CommandByte);				// Write Command to LCD
void lcd_initial(void);											// Initial LCD Nokia-5110
void lcd_clear_screen(void);									// Clear Screen Display
void lcd_fill_picture(void);									// Fill Picture Display
void lcd_gotoxy(unsigned char x,unsigned char y); 				// Set Cursor X(0..83),Y(0..5) 
void lcd_put_char(unsigned char character); 					// Put String(1 Char)
void lcd_print_string(unsigned char *string , 
                      unsigned char CharCount); 				// Print String to LCD
void Delay(vu32 nCount);

/* Private functions ---------------------------------------------------------*/

/*******************************************************************************
* Function Name  : main
* Description    : Main program.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
int main(void)
{
  
#ifdef DEBUG
  debug();
#endif

  /* System Clocks Configuration **********************************************/
  RCC_Configuration();   

  /* NVIC Configuration *******************************************************/
  NVIC_Configuration();

  /* Configure IO connected to LCD-NOKIA5110 */
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIO_SCE, ENABLE);
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
  GPIO_InitStructure.GPIO_Pin = LCD5110_SCE_PIN;
  GPIO_Init(LCD5110_SCE_PORT, &GPIO_InitStructure);

  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIO_RES, ENABLE);
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
  GPIO_InitStructure.GPIO_Pin = LCD5110_RES_PIN;
  GPIO_Init(LCD5110_RES_PORT, &GPIO_InitStructure);

  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIO_DC, ENABLE);
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
  GPIO_InitStructure.GPIO_Pin = LCD5110_DC_PIN;
  GPIO_Init(LCD5110_DC_PORT, &GPIO_InitStructure);

  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIO_LED, ENABLE);
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
  GPIO_InitStructure.GPIO_Pin = LCD5110_LED_PIN;
  GPIO_Init(LCD5110_LED_PORT, &GPIO_InitStructure);

  /* SPI2 Periph clock enable */
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_SPI2, ENABLE);

  /* Configure SPI2 pins: SCK(PB13) and MOSI(PB15) */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_13 | GPIO_Pin_15;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP;
  GPIO_Init(GPIOB, &GPIO_InitStructure);
  
  /* Config SPI[2] = Master */
  SPI_InitStructure.SPI_Direction = SPI_Direction_2Lines_FullDuplex;
  SPI_InitStructure.SPI_Mode = SPI_Mode_Master;
  SPI_InitStructure.SPI_DataSize = SPI_DataSize_8b;
  SPI_InitStructure.SPI_CPOL = SPI_CPOL_Low;
  SPI_InitStructure.SPI_CPHA = SPI_CPHA_2Edge;
  SPI_InitStructure.SPI_NSS = SPI_NSS_Soft;
  SPI_InitStructure.SPI_BaudRatePrescaler = SPI_BaudRatePrescaler_4;
  SPI_InitStructure.SPI_FirstBit = SPI_FirstBit_MSB;
  SPI_InitStructure.SPI_CRCPolynomial = 7;
  SPI_Init(SPI2, &SPI_InitStructure);
  
  /* Enable SPI2 */
  SPI_Cmd(SPI2, ENABLE);

  /* Initial GPIO Signal Interface LCD Nokia-5110 */
  LCD5110_RES_LO();												// Active Reset
  LCD5110_RES_HI();												// Normal Operation  
  LCD5110_DC_HI(); 												// D/C = High("1"=Data)
  LCD5110_LED_HI();												// LED = High(ON LED)
  LCD5110_SCE_LO();												// SCE = Low(Enable)

  /* Start Initial & Display Character to LCD */
  lcd_initial();                								// Initial LCD
  lcd_clear_screen();              								// Clear Screen Display

  // Loop Test Display //
  while(1)
  {
    lcd_gotoxy(0,0);											// Set Cursor = Line-1
    lcd_print_string("ET-STM32F103RB",14);						// Display LCD Line-1       
    lcd_print_string("BY ETT CO.,LTD",14);						// Display LCD Line-2  
    lcd_print_string("Demo SPI & LCD",14);						// Display LCD Line-3 
    lcd_print_string("LCD Nokia-5110",14);						// Display LCD Line-4  
    lcd_print_string("Demo Character",14);						// Display LCD Line-5 
    lcd_print_string("14Char x 6Line",14);						// Display LCD Line-6 	
    Delay(20000000);
  
	 lcd_fill_picture();											// Display Graphic	
    Delay(20000000);
  }  	 
}

/*******************************************************************************
* Function Name  : RCC_Configuration
* Description    : Configures the different system clocks.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void RCC_Configuration(void)
{   
  /* RCC system reset(for debug purpose) */
  RCC_DeInit();

  /* Enable HSE */
  RCC_HSEConfig(RCC_HSE_ON);

  /* Wait till HSE is ready */
  HSEStartUpStatus = RCC_WaitForHSEStartUp();

  if(HSEStartUpStatus == SUCCESS)
  {
    /* Enable Prefetch Buffer */
    FLASH_PrefetchBufferCmd(FLASH_PrefetchBuffer_Enable);

    /* Flash 2 wait state */
    FLASH_SetLatency(FLASH_Latency_2);
 	
    /* HCLK = SYSCLK */
    RCC_HCLKConfig(RCC_SYSCLK_Div1); 
  
    /* PCLK2 = HCLK */
    RCC_PCLK2Config(RCC_HCLK_Div1); 

    /* PCLK1 = HCLK/2 */
    RCC_PCLK1Config(RCC_HCLK_Div2);

    /* PLLCLK = 8MHz * 9 = 72 MHz */
    RCC_PLLConfig(RCC_PLLSource_HSE_Div1, RCC_PLLMul_9);

    /* Enable PLL */ 
    RCC_PLLCmd(ENABLE);

    /* Wait till PLL is ready */
    while(RCC_GetFlagStatus(RCC_FLAG_PLLRDY) == RESET)
    {
    }

    /* Select PLL as system clock source */
    RCC_SYSCLKConfig(RCC_SYSCLKSource_PLLCLK);

    /* Wait till PLL is used as system clock source */
    while(RCC_GetSYSCLKSource() != 0x08)
    {
    }
  }
}

/*******************************************************************************
* Function Name  : NVIC_Configuration
* Description    : Configures Vector Table base location.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_Configuration(void)
{
#ifdef  VECT_TAB_RAM  
  /* Set the Vector Table base location at 0x20000000 */ 
  NVIC_SetVectorTable(NVIC_VectTab_RAM, 0x0); 
#else  /* VECT_TAB_FLASH  */
  /* Set the Vector Table base location at 0x08000000 */ 
  NVIC_SetVectorTable(NVIC_VectTab_FLASH, 0x0);   
#endif
}

/********************************/
/* Write Data or Command to LCD */
/* D/C = "0" = Write Command    */
/* D/C = "1" = Write Display    */
/********************************/ 
void lcd_write_data(unsigned char DataByte) 
{                
  LCD5110_DC_HI(); 												// Active DC = High("1"=Data)  
  while(SPI_I2S_GetFlagStatus(SPI2, SPI_I2S_FLAG_TXE)==RESET);	// Wait for SPI2 Tx Buffer Empty   
  SPI_I2S_SendData(SPI2,DataByte);								// Send SPI2 Data 
}    

/********************************/
/* Write Data or Command to LCD */
/* D/C = "0" = Write Command    */
/* D/C = "1" = Write Display    */
/********************************/ 
void lcd_write_command(unsigned char CommandByte) 
{                  
  LCD5110_DC_LO(); 												// Active DC = Low("0"=Command)
  while(SPI_I2S_GetFlagStatus(SPI2, SPI_I2S_FLAG_TXE)==RESET);	// Wait for SPI2 Tx Buffer Empty 
  SPI_I2S_SendData(SPI2,CommandByte);							// Send SPI2 Data 
}    

/**************************/
/* Initial LCD Nokia-5110 */
/**************************/          
void lcd_initial(void)      
{   
  LCD5110_RES_LO();												// Active Reset
  LCD5110_RES_HI();												// Normal Operation

  lcd_write_command(32+1);       								// Function Set = Extend Instruction(00100+PD,V,H=00100+0,0,1)
  lcd_write_command(128+38);     								// Set VOP(1+VOP[6..0] = 1+0100110)
  lcd_write_command(4+3);            							// Temp Control(000001+TC1,TC0=000001+1,1)
  lcd_write_command(16+3);       								// Bias System(00010,BS2,BS1,BS0=00010,0,1,1)

  lcd_write_command(32+0);       								// Function Set = Basic Instruction(00100+PD,V,H = 00100+0,0,0)
  lcd_write_command(12);         								// Display Control = Normal Mode(00001D0E=00001100)
}   

/****************************/
/* Clear Screen Display LCD */
/****************************/
void lcd_clear_screen(void)       
{  
  unsigned int  i=0;   											// Memory Display(Byte) Counter
    
  lcd_write_command(128+0);       								// Set X Position = 0(0..83)
  lcd_write_command(64+0);         								// Set Y Position = 0(0..5)
  
  for(i=0;i<504;i++)   											// All Display RAM = 504 Byte  
  lcd_write_data(0);  											// Clear Screen Display
}   

/****************************/
/* Fill Picture Display LCD */
/****************************/
void lcd_fill_picture(void)       
{  
  unsigned int  i=0;   											// Memory Display(Byte) Counter
    
  lcd_write_command(128+0);       								// Set X Position = 0(0..83)
  lcd_write_command(64+0);         								// Set Y Position = 0(0..5)
  																	
  for(i=0;i<504;i++)   											// All Display RAM = 504 Byte  
  lcd_write_data(tab_picture[i]); 								// Fill Picture Display
}   

/***************************/
/* Set Cursor X,Y Position */
/* X[0-83]: 84 Column Data */
/* Y[0-5] : 6 Row(48 Dot)  */
/***************************/
void lcd_gotoxy(unsigned char x,unsigned char y)  
{  
  lcd_write_command(128+x);  	 								// Set X Position(1+x6,x5,x4,x3,x2,x1,x0)
  lcd_write_command(64+y);  									// Set Y Position(01000+y2,y1,y0)
}  

/***************************/
/* Put Char to LCD Display */
/***************************/
void lcd_put_char(unsigned char character) 
{  
  unsigned char font_size_count = 0;  							// Font Size Counter
  unsigned int  font_data_index;  								// Font Data Pointer

  font_data_index = character-32;      							// Skip 0x00..0x1F Font Code
  font_data_index = font_data_index*5;  						// 5 Byte / Font       
  
  while(font_size_count<5)                          			// Get 5 Byte Font & Display on LCD
  {  													
    lcd_write_data(tab_font[font_data_index]);    				// Get Data of Font From Table & Write LCD
    font_size_count++;  										// Next Byte Counter
    font_data_index++;  										// Next	Byte Pointer
  }  
  lcd_write_data(0);											// 1 Pixel Dot Space
}    

/*******************************/
/* Print String to LCD Display */
/*******************************/
void lcd_print_string(unsigned char *string , unsigned char CharCount) 
{          
  unsigned char i=0;  											// Dummy Character Count

  while(i<CharCount)  
  {    
    lcd_put_char(string[i]);									// Print 1-Char to LCD
    i++;                           								// Next Character Print
  }  
}

/*******************************************************************************
* Function Name  : Delay
* Description    : Inserts a delay time.
* Input          : nCount: specifies the delay time length.
* Output         : None
* Return         : None
*******************************************************************************/
void Delay(vu32 nCount)
{
  for(; nCount != 0; nCount--);
}


#ifdef  DEBUG
/*******************************************************************************
* Function Name  : assert_failed
* Description    : Reports the name of the source file and the source line number
*                  where the assert_param error has occurred.
* Input          : - file: pointer to the source file name
*                  - line: assert_param error line source number
* Output         : None
* Return         : None
*******************************************************************************/
void assert_failed(u8* file, u32 line)
{ 
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
  }
}
#endif

/******************* (C) COPYRIGHT 2008 STMicroelectronics *****END OF FILE****/
